<?php

namespace App\Http\Controllers;

use App\Models\ProductType;
use Illuminate\Http\Request;

class ProductTypeController extends Controller
{
    public function index()
    {
        $productTypes = ProductType::all();
        return view('product-types.index', compact('productTypes'));
    }

    public function create()
    {
        return view('product-types.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:product_types',
            'code' => 'required|string|max:50|unique:product_types',
            'description' => 'nullable|string',
            'default_variables' => 'nullable|array'
        ]);

        ProductType::create([
            'name' => $request->name,
            'code' => $request->code,
            'description' => $request->description,
            'default_variables' => json_encode($request->default_variables ?? [])
        ]);

        return redirect()->route('product-types.index')
            ->with('success', 'تم إضافة نوع المنتج بنجاح');
    }

    public function edit(ProductType $productType)
    {
        return view('product-types.edit', compact('productType'));
    }

    public function update(Request $request, ProductType $productType)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:product_types,name,' . $productType->id,
            'code' => 'required|string|max:50|unique:product_types,code,' . $productType->id,
            'description' => 'nullable|string',
            'default_variables' => 'nullable|array'
        ]);

        $productType->update([
            'name' => $request->name,
            'code' => $request->code,
            'description' => $request->description,
            'default_variables' => json_encode($request->default_variables ?? [])
        ]);

        return redirect()->route('product-types.index')
            ->with('success', 'تم تحديث نوع المنتج بنجاح');
    }

    public function destroy(ProductType $productType)
    {
        // التحقق من عدم وجود حسابات مرتبطة
        if ($productType->calculations()->exists()) {
            return redirect()->route('product-types.index')
                ->with('error', 'لا يمكن حذف نوع المنتج لأنه يحتوي على حسابات مرتبطة');
        }

        $productType->delete();

        return redirect()->route('product-types.index')
            ->with('success', 'تم حذف نوع المنتج بنجاح');
    }
}
